#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTest.VAL;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{
	[TestFixture]
	public class BOL_Patient : BaseTest 
	{

		private Guid _patientSInTRGuid;
		private Guid _patientGuid;
		private BOL.Patient _tstPatient;
		private DataTable _dtPatient;
		private DataTable _dtPatientSpecimen;
		private DataTable _dtBloodUnit;
		// CR2899: Need multiple units for cache testing
		private DataTable _dtBloodUnits;
		private string _patientSelectSQL = "SELECT TOP 1 *, 'B' AS " + ARTIFICIAL.ConversionBloodTypeCode + ", 'P' AS " + ARTIFICIAL.ConversionRhFactorCode + " FROM Patient WHERE PatientGuid = '{0}'";

		#region test data setup and deletion

		#endregion
		

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			if(this.RefreshData)
			{
				DataCreator.CreateCrossmatchedBloodUnitTestData();

				this._dtBloodUnit = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM BloodUnitTest WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND RecordStatusCode = 'A'", true);
				
				// CR2899
				this._dtBloodUnits = UnitTestUtil.RunSQLGetDataTable("SELECT DISTINCT TOP 3 BloodUnitGuid FROM BloodUnitTest WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND RecordStatusCode = 'A'", true);
			
				this._dtPatientSpecimen = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PatientSpecimen WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'",true);


				// Aritifical fields are needed for ConversionAbo/Rh data within BOL.Patient object
				this._dtPatient = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Patient WHERE VistaPatientId > 0 AND MergeToPatient IS NULL ORDER BY LastUpdateDate DESC", true);
				this._patientGuid = (Guid)this._dtPatient.Rows[0][TABLES.Patient.PatientGuid];

				_patientSInTRGuid = this.CreateSIandTRs();

				this.RefreshData = false;
			}

			this._tstPatient = new BOL.Patient(this._patientGuid);
		}

		[Test]
		public void Ctor_Blank_Pass()
		{
			BOL.Patient tmpPatient = new BOL.Patient();

			Assert.IsNotNull(tmpPatient);
			Assert.IsTrue(tmpPatient.IsNew);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Blank_Fail()
		{
		}

		[Test]
		public void Ctor_DataRow_Pass()
		{
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Test 1
			BOL.Patient tmpPatient = new BOL.Patient( this._dtPatient.Rows[0] );
			Assert.IsNotNull( tmpPatient );
			Assert.IsTrue( tmpPatient.PatientGuid == this._patientGuid );
			//
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Test 2: cause invalid cast exception in Dob handler
			// (for passed-in HL7 date type); this will be caught, 
			// so same assertion, also add "conversion Abo\Rh"
			DataTable dt = this._dtPatient.Copy();
			Assert.IsTrue( dt.Columns.Contains(TABLES.Patient.PatientDob) );
			//
			dt.Columns.Remove( TABLES.Patient.PatientDob );
			dt.Columns.Add( TABLES.Patient.PatientDob, typeof(string) );
			dt.Columns.Add( ARTIFICIAL.ConversionBloodTypeCode, typeof(string) );
			dt.Columns.Add( ARTIFICIAL.ConversionRhFactorCode, typeof(string) );
			dt.Rows[0][TABLES.Patient.PatientDob] = "19431225";
			dt.Rows[0][ARTIFICIAL.ConversionBloodTypeCode] = "AB";
			dt.Rows[0][ARTIFICIAL.ConversionRhFactorCode] = "P";
			//
			tmpPatient = new BOL.Patient( dt.Rows[0] );
			Assert.IsNotNull( tmpPatient );
			Assert.IsTrue( tmpPatient.PatientGuid == this._patientGuid );
			//
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Test 3: Incorrect Rh Factor Code; this will be caught, 
			// so same assertion
			dt = this._dtPatient.Copy(); 
			dt.Columns.Add( ARTIFICIAL.ConversionBloodTypeCode, typeof(string) );
			dt.Columns.Add( ARTIFICIAL.ConversionRhFactorCode, typeof(string) );
			dt.Rows[0][ARTIFICIAL.ConversionBloodTypeCode] = "A";
			dt.Rows[0][ARTIFICIAL.ConversionRhFactorCode] = "ZZ";
			//
			tmpPatient = new BOL.Patient( dt.Rows[0] );
			Assert.IsNotNull( tmpPatient );
			Assert.IsTrue( tmpPatient.PatientGuid == this._patientGuid );
		}

		[Test]
		public void Ctor_DataRowMerge_Pass()
		{
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Test 1
			this._dtPatient.Rows[0][Common.VbecsTables.Patient.PatientDeathDate] = DateTime.Now;
			this._dtPatient.Rows[0][Common.VbecsTables.Patient.FinalMergeToPatient] = Guid.NewGuid();
			BOL.Patient tmpPatient = new BOL.Patient( this._dtPatient.Rows[0] );
			Assert.IsNotNull( tmpPatient );
			Assert.IsTrue( tmpPatient.PatientGuid == this._patientGuid );
			//
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Test 2: cause invalid cast exception in Dob handler
			// (for passed-in HL7 date type); this will be caught, 
			// so same assertion, also add "conversion Abo\Rh"
			DataTable dt = this._dtPatient.Copy();
			Assert.IsTrue( dt.Columns.Contains(TABLES.Patient.PatientDob) );
			//
			dt.Columns.Remove( TABLES.Patient.PatientDob );
			dt.Columns.Add( TABLES.Patient.PatientDob, typeof(string) );
			dt.Columns.Add( ARTIFICIAL.ConversionBloodTypeCode, typeof(string) );
			dt.Columns.Add( ARTIFICIAL.ConversionRhFactorCode, typeof(string) );
			dt.Rows[0][TABLES.Patient.PatientDob] = "19431225";
			dt.Rows[0][ARTIFICIAL.ConversionBloodTypeCode] = "AB";
			dt.Rows[0][ARTIFICIAL.ConversionRhFactorCode] = "P";
			//
			tmpPatient = new BOL.Patient( dt.Rows[0] );
			Assert.IsNotNull( tmpPatient );
			Assert.IsTrue( tmpPatient.PatientGuid == this._patientGuid );
			//
			// ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
			// Test 3: Incorrect Rh Factor Code; this will be caught, 
			// so same assertion
			dt = this._dtPatient.Copy(); 
			dt.Columns.Add( ARTIFICIAL.ConversionBloodTypeCode, typeof(string) );
			dt.Columns.Add( ARTIFICIAL.ConversionRhFactorCode, typeof(string) );
			dt.Rows[0][ARTIFICIAL.ConversionBloodTypeCode] = "A";
			dt.Rows[0][ARTIFICIAL.ConversionRhFactorCode] = "ZZ";
			//
			tmpPatient = new BOL.Patient( dt.Rows[0] );
			Assert.IsNotNull( tmpPatient );
			Assert.IsTrue( tmpPatient.PatientGuid == this._patientGuid );
		}

		[Test]
		[ExpectedException(typeof(FormatException))]
		public void Ctor_DataRow_Fail()
		{
			BOL.Patient tstValue = new BOL.Patient(this._dtPatient.NewRow());
		}

		[Test]
		public void Ctor_DataRow_Bool_Pass()
		{
			BOL.Patient tmpPatient = new BOL.Patient(this._dtPatient.Rows[0], true);

			Assert.IsNotNull(tmpPatient);
			Assert.IsTrue(tmpPatient.PatientGuid == this._patientGuid);
			Assert.IsTrue(tmpPatient.ImportedFromVista);
		}

		[Test]
		public void Ctor_DataRow_Bool_Fail()
		{
			DataRow drTemp = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM Patient");
			Guid patientGuid = (Guid)drTemp[TABLES.Patient.PatientGuid];
			
			BOL.Patient tmpPatient = new BOL.Patient(drTemp, false);
			
			Assert.AreEqual(patientGuid, tmpPatient.PatientGuid, "PatientGuid");
			Assert.AreEqual(false, tmpPatient.ImportedFromVista, "ImportedFromVista");
		}
		
		
		[Test]
		public void Ctor_Guid_Pass()
		{
			BOL.Patient tmpPatient = new BOL.Patient(this._patientGuid);

			Assert.IsNotNull(tmpPatient);
			Assert.IsTrue(tmpPatient.PatientGuid == this._patientGuid);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Ctor_Guid_Fail()
		{
		}


		[Test]
		public void PatientGuid_Pass()
		{
			Guid tstValue = Guid.NewGuid();
			
			this._tstPatient.PatientGuid = tstValue;

			Assert.IsTrue(this._tstPatient.PatientGuid == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PatientGuid_Fail()
		{
		}

		[Test]
		public void PatientMergeGroup_Pass()
		{
			Guid tstValue = Guid.NewGuid();
				
			this._tstPatient.PatientMergeGroup = tstValue;

			Assert.AreEqual(tstValue, this._tstPatient.PatientMergeGroup);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PatientMergeGroup_Fail()
		{
		}

		[Test]
		public void FinalMergeToPatient_Pass()
		{
			Guid tstValue = Guid.NewGuid();
				
			this._tstPatient.FinalMergeToPatient = tstValue;

			Assert.AreEqual(tstValue, this._tstPatient.FinalMergeToPatient);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void FinalMergeToPatient_Fail()
		{
		}

		[Test]
		public void DisplayName_Pass()
		{
			string tstValue = this._tstPatient.DisplayName;
			Assert.IsTrue(this._tstPatient.DisplayName == tstValue);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void DisplayName_Fail()
		{
		}

		[Test]
		public void PatientDobCode_Pass()
		{
			string tstValue = DateTime.Now.ToString();
			
			this._tstPatient.PatientDobCode = tstValue;

			Assert.IsTrue(this._tstPatient.PatientDobCode == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PatientDobCode_Fail()
		{
		}

		[Test]
		public void ValidPatientDOB_Pass()
		{
			this._tstPatient.PatientDobCode = "B";
			Assert.IsTrue(this._tstPatient.ValidPatientDOB == false);
			this._tstPatient.PatientDobCode = "D";
			Assert.IsTrue(this._tstPatient.ValidPatientDOB == false);
			this._tstPatient.PatientDobCode = "M";
			Assert.IsTrue(this._tstPatient.ValidPatientDOB == false);
			this._tstPatient.PatientDobCode = "A";
			Assert.IsTrue(this._tstPatient.ValidPatientDOB == true);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ValidPatientDOB_Fail()
		{
		}

		[Test]
		public void FirstName_Pass()
		{
			string tstValue = "".PadRight(30,'X');

			this._tstPatient.FirstName = tstValue;

			Assert.IsTrue(this._tstPatient.FirstName == tstValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void FirstName_Fail()
		{
			string tstValue = "".PadRight(31, 'X');
			
			this._tstPatient.FirstName = tstValue;
		
		}

		[Test]
		public void LastName_Pass()
		{
			string tstValue = "".PadRight(30,'X');

			this._tstPatient.LastName = tstValue;

			Assert.IsTrue(this._tstPatient.LastName == tstValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void LastName_Fail()
		{
			string tstValue = "".PadRight(31,'X');
			
			this._tstPatient.LastName = tstValue;
		}

		[Test]
		public void MiddleInitial_Pass()
		{
			string tstValue = DateTime.Now.Date.ToString();

			this._tstPatient.MiddleInitial = tstValue;

			Assert.IsTrue(this._tstPatient.MiddleInitial == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void MiddleInitial_Fail()
		{
		}

		
		[Test]
		public void ImportedFromVista_Pass()
		{
			bool tstValue = (!this._tstPatient.ImportedFromVista);

			this._tstPatient.ImportedFromVista = tstValue;

			Assert.IsTrue(this._tstPatient.ImportedFromVista == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ImportedFromVista_Fail()
		{
		}
		
		[Test]
		public void SSN_Pass()
		{
			string tstValue = new Random().Next(111111111,999999999).ToString();

			this._tstPatient.SSN = tstValue;

			Assert.IsTrue(this._tstPatient.SSN == tstValue);
		}
		
		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void SSN_Fail()
		{
			this._tstPatient.SSN = "".PadRight(10,'X');
		}
		
		[Test]
		public void ABORhSource_Pass()
		{
			Common.ABORhSource tstValue = this._tstPatient.ABORhSource;

			Assert.IsTrue(this._tstPatient.ABORhSource == tstValue);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void ABORhSource_Fail()
		{
		}
		
		[Test]
		public void DisplayPatientID_Pass()
		{
			string tstValue = this._tstPatient.DisplayPatientID;

			Assert.IsTrue(this._tstPatient.DisplayPatientID == tstValue);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void DisplayPatientID_Fail()
		{
		}
		
		[Test]
		public void DateOfBirth_Pass()
		{
			DateTime tstValue = DateTime.Now.AddYears(-1);

			this._tstPatient.DateOfBirth = tstValue;
			Assert.IsTrue(this._tstPatient.DateOfBirth == tstValue);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(BOL.BusinessObjectException))]
		public void DateOfBirth_Fail()
		{
			DateTime tstValue = DateTime.Now.AddDays(1);

			this._tstPatient.DateOfBirth = tstValue;
		}
		
		[Test]
		public void SpecimenUid_Pass()
		{
			string tstValue = "".PadRight(20,'X');

			this._tstPatient.SpecimenUid = tstValue;
			Assert.IsTrue(this._tstPatient.SpecimenUid == tstValue);
		}
		
		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void SpecimenUid_Fail()
		{
			string tstValue = "".PadRight(21,'X');

			this._tstPatient.SpecimenUid = tstValue;
		}

		
		[Test]
		public void Sex_Pass()
		{
			Common.Sex tstValue = Common.Sex.Male;
			this._tstPatient.Sex = tstValue;
			Assert.IsTrue(this._tstPatient.Sex == tstValue);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Sex_Fail()
		{
		}

		[Test]
		public void SpecialInstructions_Pass()
		{
			DataTable dtTest = this._tstPatient.SpecialInstructions;
			Assert.IsTrue(dtTest.Rows.Count >= 0);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SpecialInstructions_Fail()
		{
		}
		
		[Test]
		public void TransfusionRequirements_Pass()
		{
			DataTable dtTest = this._tstPatient.TransfusionRequirements;
			Assert.IsTrue(dtTest.Rows.Count >= 0);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void TransfusionRequirements_Fail()
		{
		}
		
		[Test]
		public void ClinicallySignificantAntibodies_Pass()
		{
			DataTable dtTest = this._tstPatient.ClinicallySignificantAntibodies;
			Assert.IsTrue(dtTest.Rows.Count >= 0);
		}

		[Test]
		public void ClinicallySignificantAntibodies_TRsFromConversion_Pass()
		{
			Guid TRGuid = Guid.NewGuid();
			Guid patientGuid = this._tstPatient.PatientGuid;

			string insertTransfusionReq = "INSERT INTO PATIENTTRANSFUSIONREQUIREMENT (";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDivisionCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			insertTransfusionReq += "VALUES (";
			insertTransfusionReq += "'" + TRGuid.ToString() + "',";
			insertTransfusionReq += "'" + patientGuid.ToString() + "',";
			insertTransfusionReq += "'Anti-C',";
			insertTransfusionReq += "'AS',";
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertTransfusionReq += "'1',";
			insertTransfusionReq += "'5',";
			insertTransfusionReq += "'C',";  //TR from conversion
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertTransfusionReq);
		
			DataTable dtTest = this._tstPatient.ClinicallySignificantAntibodies;
			for (int i=0;i<dtTest.Rows.Count;i++)
				if (Convert.ToInt32(dtTest.Rows[0][Common.VbecsTables.PatientTransfusionRequirement.AntibodyTypeId]) == 5)
				{
					Assert.IsTrue(true);
					return;
				}
			Assert.IsTrue(false);
		}
		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ClinicallySignificantAntibodies_Fail()
		{
		}
		
		
		[Test]
		public void VistaPatientID_Pass()
		{
			long tstValue = (long) new Random().Next(1, 999999999) * 100;

			this._tstPatient.VistaPatientId = tstValue;
			Assert.IsTrue(this._tstPatient.VistaPatientId == tstValue);
		}
		
		[Test]
		[ExpectedExceptionAttribute(typeof(BusinessObjectException))]
		public void VistaPatientID_Fail()
		{
			this._tstPatient.VistaPatientId = 0;
		}

		[Test]
		public void CompatibilityPercentage_Pass()
		{
			string perc = this._tstPatient.CompatibilityPercentage;
			Assert.AreEqual(perc,_tstPatient.CompatibilityPercentage,"CompatibilityPercentage");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void CompatibilityPercentage_Fail()
		{
		}

		[Test]
		public void AboRh_Pass()
		{
			BOL.AboRh tstValue = this._tstPatient.AboRh;
			Assert.IsTrue(this._tstPatient.AboRh == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void AboRh_Fail()
		{
		}

		[Test]
		public void ConversionAboRH_Pass()
		{
			BOL.AboRh tstValue = this._tstPatient.ConversionAboRH;

			Assert.IsTrue(this._tstPatient.ConversionAboRH == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void ConversionAboRH_Fail()
		{
		}

		[Test]
		public void PreviousAboRH_Pass()
		{
			BOL.AboRh tstValue = this._tstPatient.PreviousAboRH;
			Assert.IsTrue(this._tstPatient.PreviousAboRH.AboRHText == tstValue.AboRHText);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void PreviousAboRH_Fail()
		{
		}

		[Test]
		public void PatientServiceBranch_Pass()
		{
			string tstValue = DateTime.Now.ToString();

			this._tstPatient.PatientServiceBranch = tstValue;
			Assert.IsTrue(this._tstPatient.PatientServiceBranch == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void PatientServiceBranch_Fail()
		{
		}

		
		[Test]
		public void ValidSpecimenExists_Pass()
		{
			bool tstValue = this._tstPatient.ValidSpecimenExists;

			Assert.IsTrue(this._tstPatient.ValidSpecimenExists == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void ValidSpecimenExists_Fail()
		{
		}

		[Test]
		public void ElectronicCrossmatchIndicator_Pass()
		{
			bool tstValue = this._tstPatient.ElectronicCrossmatchIndicator;

			Assert.IsTrue(this._tstPatient.ElectronicCrossmatchIndicator == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void ElectronicCrossmatchIndicator_Fail()
		{
		}

		[Test]
		public void AboRHHistory_Pass()
		{
			DataTable tstValue = this._tstPatient.AboRHHistory;

			Assert.AreEqual(this._tstPatient.AboRHHistory.Rows.Count, tstValue.Rows.Count, "Rows.Count");
			Assert.AreEqual(this._tstPatient.AboRHHistory.Columns.Count, tstValue.Columns.Count, "Columns.Count");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void AboRHHistory_Fail()
		{
		}


		[Test]
		public void JustifiedAboRH_Pass()
		{
			bool tstValue = this._tstPatient.JustifiedAboRH;

			Assert.IsTrue(this._tstPatient.JustifiedAboRH == tstValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void JustifiedAboRH_Fail()
		{
		}

		
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]			//Ignore Attribute added -- see notes in method
		public void DetermineElectronicCrossmatchEligibility_Pass()
		{
			// These are all covered in the PatientUnitSelection unit tests
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_Pass();
		}

		[Ignore(StandardIgnoreReasons.NoFailureConditions)]		//Ignore Attribute added -- see notes in method
		public void DetermineElectronicCrossmatchEligibility_Fail()
		{
			
			// These are all covered in the PatientUnitSelection unit tests
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_ABORhHistory_Discrepancy_Fail();
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_ABORhHistory_NotEnoughTypings_Fail();
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_AntibodyIndentified_Fail();
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_DivisionNotEnabled_Fail();
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_NoSpecimenResults_Fail();
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_TransfusionRequirement_AntibodyHistory_Fail();
			// See -- UnitTests.PatientUnitSelection_BOL().GetDivisionAndPatientEXMStatus_TransfusionRequirement_AntigenNegative_Fail();
		}

		[Test]
		public void GetPatientHistoricABORhForTooltip_Pass()
		{
			// first, select a patient with test results (ABO/Rh will not be NA)
			string sqlPatientWithTestResults = "select TOP 1 p.patientguid from patient p join patientspecimen ps on p.patientguid = ps.patientguid join specimentest st on ps.patientspecimenguid = st.patientspecimenguid where (bloodtesttypeid = 18 and st.recordstatuscode = 'A') or (bloodtesttypeid = 65 and st.recordstatuscode = 'A') order by ps.lastupdatedate desc";

			DataRow drPatient = UnitTestUtil.RunSQLGetDataRow(sqlPatientWithTestResults);
			BOL.Patient patient = new BOL.Patient( (Guid) drPatient[TABLES.Patient.PatientGuid]);

			string aboRh = patient.GetPatientHistoricABORhForTooltip();
			Assert.IsTrue(aboRh.Length > 0,"Historic ABORh");
			
			// next, test with a patient with no test results
			string sqlPatientWithNoTestResults = "select top 1 p.patientguid from patient p where mergetopatient is null and (select count(*) from patientspecimen ps where p.patientguid = ps.patientguid) = 0 and bloodtypecode is null and rhfactorcode is null order by LastUpdateDate desc";

			DataRow drPatientNoAboRh = UnitTestUtil.RunSQLGetDataRow(sqlPatientWithNoTestResults);
			BOL.Patient patientNoAboRh = new BOL.Patient( (Guid) drPatientNoAboRh[TABLES.Patient.PatientGuid]);

			aboRh = patientNoAboRh.GetPatientHistoricABORhForTooltip();
			Assert.IsTrue(aboRh.Length > 0 && aboRh == "There are no ABO/Rh test results on file.","No ABORh");

			// Now, update a record on patient: A Neg
			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = 'A', RhFactorCode = 'N' WHERE PatientGuid = '" + patientNoAboRh.PatientGuid + "'");

			aboRh = patientNoAboRh.GetPatientHistoricABORhForTooltip();
			Assert.IsTrue(aboRh.Length > 0 && aboRh == "From conversion");


			// Now, update a record on patient: A Pos
			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = 'A', RhFactorCode = 'P' WHERE PatientGuid = '" + patientNoAboRh.PatientGuid + "'");

			aboRh = patientNoAboRh.GetPatientHistoricABORhForTooltip();
			Assert.IsTrue(aboRh.Length > 0 && aboRh == "From conversion");


			// Now, update a record on patient: A NA
			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = 'A', RhFactorCode = '' WHERE PatientGuid = '" + patientNoAboRh.PatientGuid + "'");

			aboRh = patientNoAboRh.GetPatientHistoricABORhForTooltip();
			Assert.IsTrue(aboRh.Length > 0 && aboRh == "From conversion");


			// Set back
			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = NULL, RhFactorCode = NULL WHERE PatientGuid = '" + patientNoAboRh.PatientGuid + "'");
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetPatientHistoricABORhForTooltip_Fail()
		{
		}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			// We're doing this a bit different to get coverage on the unused Merge stuff
			DataRow dr = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM Patient");
			Guid patientGuid = (Guid)dr["PatientGuid"];
			Guid fakeMergeToPatient = 
				(Guid)UnitTestUtil.RunSQLGetDataRow(string.Concat("SELECT TOP 1 PatientGuid FROM Patient WHERE PatientGuid <> '", patientGuid, "'"))["PatientGuid"];
			//
			UnitTestUtil.RunSQL(string.Concat(
				"UPDATE Patient SET PatientMergeGroup = '", Guid.NewGuid(), "', FinalMergeToPatient = '", fakeMergeToPatient, "' WHERE PatientGuid = '", patientGuid, "'"));
			//
			BOL.Patient tempPatient = new Patient(patientGuid);
			//
			dr = UnitTestUtil.RunSQLGetDataRow(string.Concat("SELECT * FROM Patient WHERE PatientGuid = '", patientGuid, "'"));
			//
			tempPatient = new Patient(dr);
			//
			Assert.IsTrue(tempPatient != null);
			//
			dr = new DataTable().NewRow();
			//
			Assert.IsTrue(tempPatient.LoadDataRowFromThis(dr) == dr);
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void LoadDataRowFromThis_Fail()
		{
		}


		[Test]
		public void Clear_Pass()
		{
			this._tstPatient.Clear();

			Assert.AreEqual(Guid.Empty,this._tstPatient.PatientGuid);
			Assert.AreEqual("",this._tstPatient.LastName);
			Assert.AreEqual("",this._tstPatient.FirstName);
			Assert.AreEqual("",this._tstPatient.MiddleInitial);
			Assert.AreEqual(Common.ABO.NA,this._tstPatient.AboRh.Abo);
			Assert.AreEqual(Common.RH.NotProvided,this._tstPatient.AboRh.RH);
			Assert.AreEqual("",this._tstPatient.SSN);
			Assert.AreEqual(Common.Sex.Unknown,this._tstPatient.Sex);
			Assert.AreEqual(long.MinValue,this._tstPatient.VistaPatientId);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Clear_Fail()
		{
		}

		[Test]
		public void GetPatientAboRHHistory_Pass()
		{
			//Insert Patient ABO/Rh results to ensure success.
			BOL.UnitTests.BOL_SpecimenTest.BuildABORhFromSpecimenTests(this._tstPatient.PatientGuid, Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),Guid.NewGuid(),"A","P");
			//
			DataTable dtTest = this._tstPatient.GetPatientAboRHHistory();
			Assert.IsTrue(dtTest.Rows.Count > 0);
		}

		
		[Test]
		public void GetPatientAboRHHistory_Fail()
		{
			BOL.Patient patient = new BOL.Patient();
			//
			DataTable dtTest = patient.GetPatientAboRHHistory();
			Assert.IsTrue(dtTest.Rows.Count == 0);
		}

		[Test]
		public void ValidateAntibodyAntigenRequirementsMet_Pass()
		{
			Guid unitGuid = (Guid) _dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid);

			// First, test with a Neg unit to Anti-D is ignored
			UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET LastUpdateDate = GETUTCDATE(), RhFactorCode = 'N' WHERE BloodUnitGuid = '" + unitGuid + "'");
			// Unit cannot be autologous
			UnitTestUtil.RunSQL("UPDATE BloodUnit SET DonationTypeId = 2 WHERE BloodUnitGuid = '" + unitGuid + "'");


			tmpUnit.BloodUnitMedia.AboRh.RH = Common.RH.Negative;
			DeleteConflictingData(_tstPatient, tmpUnit, true, true, true);

			// Adding twice to make sure duplicates are ignored.
			SetupAntigenTr(_tstPatient, unitGuid);
			SetupAntigenTr(_tstPatient, unitGuid);

			// Must recreate patient because of caching
			_tstPatient = new BOL.Patient(_tstPatient.PatientGuid);
			System.Collections.ArrayList tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);


			// Testing condition #1a - antigen with a Neg unit
			Assert.IsTrue(tstValue.Count == 1);
			BOL.AntigenNegativeMessage aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6);


			// Next, make the unit Positive so Anti-D is picked up
            UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET LastUpdateDate = GETUTCDATE(), RhFactorCode = 'P' WHERE BloodUnitGuid = '" + unitGuid + "'");
			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);


			// Testing condition #1b - antigen with a Pos unit
			Assert.IsTrue(tstValue.Count == 2);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6 || aNM.AntigenTypeId == 15);

			aNM = (BOL.AntigenNegativeMessage) tstValue[1];
			Assert.IsTrue(aNM.AntigenTypeId == 6 || aNM.AntigenTypeId == 15);




			// Next up, add an antibody
			SetupAntibodyTr(_tstPatient, unitGuid);

			// Must recreate patient because of caching
			_tstPatient = new BOL.Patient(_tstPatient.PatientGuid);
			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);


			// Testing condition #2 - antibody
			Assert.IsTrue(tstValue.Count == 3);
			aNM = (BOL.AntigenNegativeMessage) tstValue[2];
			Assert.IsTrue(aNM.AntigenTypeId == 20);






			// Testing condition #3, antigens and antibodies
			// So, what we're doing here is testing the unit negative for C and Anti-E.
			// Since the unit will be negative for these, only
			// 1 result will be returned (AN: 15 since unit is Rh positive)
			// Patient: 
			//	C: Pos
			//  D: Pos
			//  Anti-E : Pos
			SetupUnitTest(tmpUnit, "0", 6);
			SetupUnitTest(tmpUnit, "0", 20);

			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);
			Assert.IsTrue(tstValue.Count == 1);







			// Testing condition #4, antigens and antibodies
			// So, what we're doing here is entering positive test results for the 
			// units.  
			DeleteConflictingData(_tstPatient, tmpUnit, true, false, false);
			SetupUnitTest(tmpUnit, "1", 6);
			SetupUnitTest(tmpUnit, "1", 20);

			// Must recreate patient because of caching
			_tstPatient = new BOL.Patient(_tstPatient.PatientGuid);

			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);
			Assert.IsTrue(tstValue.Count == 3);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6  || aNM.AntigenTypeId == 15 || aNM.AntigenTypeId == 20);
			aNM = (BOL.AntigenNegativeMessage) tstValue[1];
			Assert.IsTrue(aNM.AntigenTypeId == 6  || aNM.AntigenTypeId == 15 || aNM.AntigenTypeId == 20);
			aNM = (BOL.AntigenNegativeMessage) tstValue[2];
			Assert.IsTrue(aNM.AntigenTypeId == 6  || aNM.AntigenTypeId == 15 || aNM.AntigenTypeId == 20);
		
			DeleteConflictingData(_tstPatient, tmpUnit, true, false, false);





			// Testing condition #5, antigens and antibodies
			SetupBloodUnitTests(unitGuid, 212, "N");
			SetupBloodUnitTests(unitGuid, 218, "N");

			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);
			Assert.IsTrue(tstValue.Count == 1);
			
			DeleteConflictingData(_tstPatient, tmpUnit, false, false, true);



			// Testing condition #6, antigens and antibodies
			SetupBloodUnitTests(unitGuid, 212, "I");
			SetupBloodUnitTests(unitGuid, 218, "I");
			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);
			Assert.IsTrue(tstValue.Count == 3);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6  || aNM.AntigenTypeId == 15 || aNM.AntigenTypeId == 20);
			aNM = (BOL.AntigenNegativeMessage) tstValue[1];
			Assert.IsTrue(aNM.AntigenTypeId == 6  || aNM.AntigenTypeId == 15 || aNM.AntigenTypeId == 20);
			aNM = (BOL.AntigenNegativeMessage) tstValue[2];
			Assert.IsTrue(aNM.AntigenTypeId == 6  || aNM.AntigenTypeId == 15 || aNM.AntigenTypeId == 20);





			// Finally, enter duplicate data and make sure duplicate results aren't returned 
			SetupAntigenTr(_tstPatient, unitGuid);
			SetupAntibodyTr(_tstPatient, unitGuid);
			tstValue = _tstPatient.ValidateAntibodyAntigenRequirementsMet(tmpUnit);
			Assert.IsTrue(tstValue.Count == 3);

			// Clean up data
			DeleteConflictingData(_tstPatient, tmpUnit, true, true, true);
		}



		// CR2899: The ValidateAntibodyAntigenRequirementsMet_Pass/_Fail methods are already
		// testing the antigen/antibody functionality in the ValidateAntibodyAntigenRequirements
		// method.  The purpose of these tests is to test the new code that allows multiple 
		// units to be passed in at once to facilitate caching.
		[Test]
		public void ValidateAntibodyAntigenRequirementsMetCache_Pass()
		{
			Guid unitGuid1 = (Guid) _dtBloodUnits.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
			Guid unitGuid2 = (Guid) _dtBloodUnits.Rows[1][TABLES.BloodUnit.BloodUnitGuid];
			Guid unitGuid3 = (Guid) _dtBloodUnits.Rows[2][TABLES.BloodUnit.BloodUnitGuid];


			BOL.BloodUnit tmpUnit1 = new BOL.BloodUnit(unitGuid1);
			BOL.BloodUnit tmpUnit2 = new BOL.BloodUnit(unitGuid2);
			BOL.BloodUnit tmpUnit3 = new BOL.BloodUnit(unitGuid3);



			// Make the first unit Positive, returning a positive result for Anti-D
            UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET LastUpdateDate = GETUTCDATE(), RhFactorCode = 'P' WHERE BloodUnitGuid = '" + unitGuid1 + "'");
			// Make the other two Negative, returning a negative result for Anti-D
            UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET LastUpdateDate = GETUTCDATE(), RhFactorCode = 'N' WHERE BloodUnitGuid = '" + unitGuid2 + "'");
            UnitTestUtil.RunSQL("UPDATE BloodUnitMedia SET LastUpdateDate = GETUTCDATE(), RhFactorCode = 'N' WHERE BloodUnitGuid = '" + unitGuid3 + "'");
	

			DeleteConflictingData(_tstPatient, tmpUnit1, true, true, true);
			DeleteConflictingData(_tstPatient, tmpUnit2, true, true, true);
			DeleteConflictingData(_tstPatient, tmpUnit3, true, true, true);

			ArrayList unitsAL = new ArrayList(3);
			unitsAL.Add(tmpUnit1);
			unitsAL.Add(tmpUnit2);
			unitsAL.Add(tmpUnit3);

			
			
			//--->>> Patient will be positive for antigens C and D
			SetupAntigenTr(_tstPatient, unitGuid1);

			// Must recreate patient because of caching
			_tstPatient = new BOL.Patient(_tstPatient.PatientGuid);

			Hashtable htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);
			
			// Make sure 3 results come back since 3 units were passed in.
			Assert.IsTrue(htResults.Count == 3);

		
			// Testing condition #1 - antigen
			// Get the results for the first blood unit
			ArrayList tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 2);
			BOL.AntigenNegativeMessage aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6 || aNM.AntigenTypeId == 15);
			aNM = (BOL.AntigenNegativeMessage) tstValue[1];
			Assert.IsTrue(aNM.AntigenTypeId == 6 || aNM.AntigenTypeId == 15);

			// Get the results for the second blood unit
			tstValue = (ArrayList) htResults[tmpUnit2.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 1);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6);

			// Get the results for the third blood unit
			tstValue = (ArrayList) htResults[tmpUnit2.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 1);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6);
			




			
			// Patient is now positive for Anti-E (in addition to previously set up C and D)
			SetupAntibodyTr(_tstPatient, unitGuid1);
			this._tstPatient = new BOL.Patient(this._patientGuid); // do this to renew cache objects
			htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);
			
			
			// Testing condition #1 - antibody
			// Unit #1
			tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];	
			Assert.IsTrue(tstValue.Count == 3);
			aNM = (BOL.AntigenNegativeMessage) tstValue[2];
			Assert.IsTrue(aNM.AntigenTypeId == 20);

			// Testing condition #1 - antibody
			// Unit #2
			tstValue = (ArrayList) htResults[tmpUnit2.BloodUnitGuid];	
			Assert.IsTrue(tstValue.Count == 2);
		
			// Testing condition #1 - antibody
			// Unit #3
			tstValue = (ArrayList) htResults[tmpUnit3.BloodUnitGuid];	
			Assert.IsTrue(tstValue.Count == 2);
	




			// Testing condition #2, antigens and antibodies
			// Unit #1 is negative for C and Anti-E
			SetupUnitTest(tmpUnit1, "0", 6);
			SetupUnitTest(tmpUnit1, "0", 20);

			// Unit #2 is negative for C
			SetupUnitTest(tmpUnit2, "0", 6);

			// Unit #3 is negative for Anti-E
			SetupUnitTest(tmpUnit3, "0", 20);

			htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);

			// Unit #1
			tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 1);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 15);

			// Unit #2
			tstValue = (ArrayList) htResults[tmpUnit2.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 1);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 20);

			// Unit #3
			tstValue = (ArrayList) htResults[tmpUnit3.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 1);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6);








			// Testing condition #4, antigens and antibodies
			DeleteConflictingData(_tstPatient, tmpUnit1, true, false, false);
			DeleteConflictingData(_tstPatient, tmpUnit2, true, false, false);
			DeleteConflictingData(_tstPatient, tmpUnit3, true, false, false);

			// Unit #1 is positive for C and Anti-E
			SetupUnitTest(tmpUnit1, "1", 6);
			SetupUnitTest(tmpUnit1, "1", 20);

			// Unit #2 is positive for Anti-E
			SetupUnitTest(tmpUnit2, "1", 20);

			// Unit #3 is positive for C
			SetupUnitTest(tmpUnit3, "1", 6);


			htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);

			// Unit #1
			tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 3);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			Assert.IsTrue(aNM.AntigenTypeId == 6 || aNM.AntigenTypeId == 15);
			aNM = (BOL.AntigenNegativeMessage) tstValue[1];
			Assert.IsTrue(aNM.AntigenTypeId == 6 || aNM.AntigenTypeId == 15);
			aNM = (BOL.AntigenNegativeMessage) tstValue[2];
			Assert.IsTrue(aNM.AntigenTypeId == 20);

			// Unit #2
			tstValue = (ArrayList) htResults[tmpUnit2.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 2);
		

			// Unit #3
			tstValue = (ArrayList) htResults[tmpUnit3.BloodUnitGuid];
			Assert.IsTrue(tstValue.Count == 2);
			

		



			


			// Testing condition #4, antigens and antibodies
			DeleteConflictingData(_tstPatient, tmpUnit1, true, false, false);
			DeleteConflictingData(_tstPatient, tmpUnit2, true, false, false);
			DeleteConflictingData(_tstPatient, tmpUnit3, true, false, false);

			SetupBloodUnitTests(unitGuid1, 212, "N");
			SetupBloodUnitTests(unitGuid1, 218, "N");
			htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);
			tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];
	
			Assert.IsTrue(tstValue.Count == 1);
			
			DeleteConflictingData(_tstPatient, tmpUnit1, false, false, true);



			// Testing condition #5, antigens and antibodies
			SetupBloodUnitTests(unitGuid1, 212, "I");
			SetupBloodUnitTests(unitGuid1, 218, "I");
			htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);
			tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];
	
			Assert.IsTrue(tstValue.Count == 3);
			aNM = (BOL.AntigenNegativeMessage) tstValue[0];
			





			// Finally, enter duplicate data and make sure duplicate results aren't returned 
			SetupAntigenTr(_tstPatient, unitGuid1);
			SetupAntibodyTr(_tstPatient, unitGuid1);
			htResults = _tstPatient.ValidateAntibodyAntigenRequirementsMet(unitsAL);
			tstValue = (ArrayList) htResults[tmpUnit1.BloodUnitGuid];
			
			Assert.IsTrue(tstValue.Count == 3);

			// Clean up data
			DeleteConflictingData(_tstPatient, tmpUnit1, true, true, true);
		}


		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void ValidateAntibodyAntigenRequirementsMetCache_Fail()
		{
			Hashtable results = _tstPatient.ValidateAntibodyAntigenRequirementsMet(new ArrayList());
		}

		// CR2899: The ValidateComponentRequirementsMet_Pass/_Fail methods are already
		// testing the component functionality in the ValidateComponentRequirements method.
		// The purpose of these tests is to test the new code that allows multiple units
		// to be passed in at once to facilitate caching.
		[Test]
		public void ValidateComponentRequirementsCache_Pass()
		{	
			Guid bloodUnitGuid1 = Guid.NewGuid();
			Guid bloodUnitGuid2 = Guid.NewGuid();


			BOL.Patient patient = new Patient(_patientGuid);
			
			// These unit will satisfy all of the component requirements.
			DataRow drProduct = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E3571'");
			
			DataCreator.CreateBloodUnit(bloodUnitGuid1,Guid.NewGuid(),_patientGuid,(Guid)drProduct[TABLES.BloodProduct.BloodProductGuid], false);
			DataCreator.CreateBloodUnit(bloodUnitGuid2,Guid.NewGuid(),_patientGuid,(Guid)drProduct[TABLES.BloodProduct.BloodProductGuid], false);
			
			// Create all component requirements for the patient
			this.CreateAllCRs(patient);

			ArrayList units = new ArrayList(2);
			units.Add( new BOL.BloodUnit(bloodUnitGuid1) );
			units.Add( new BOL.BloodUnit(bloodUnitGuid2) );

			Hashtable results = patient.ValidateComponentRequirements(units);
			
			// Two arraylists are returned, one for each unit
			Assert.IsTrue(results.Count == 2);


			// Results: These units satisfied the component requirements, so no results returned.
			ArrayList compResults;
			compResults = (ArrayList) results[bloodUnitGuid1];
			Assert.IsTrue(compResults.Count == 0);

			compResults = (ArrayList) results[bloodUnitGuid2];
			Assert.IsTrue(compResults.Count == 0);
			
			// Clean up
			this.DeleteAllCRs(patient);
		}



		// CR2899: The ValidateComponentRequirementsMet_Pass/_Fail methods are already
		// testing the component functionality in the ValidateComponentRequirements method.
		// The purpose of these tests is to test the new code that allows multiple units
		// to be passed in at once to facilitate caching.
		[Test]
		public void ValidateComponentRequirementsCache_Fail()
		{
			// So, the goal in thies test is to create 2 units, one that will satisfy the requirement and
			// one that will not.  This will make sure the method is processing/sorting them correctly.

			Guid bloodUnitGuid1 = Guid.NewGuid();
			Guid bloodUnitGuid2 = Guid.NewGuid();
			Guid bloodUnitGuid3 = Guid.NewGuid();

			BOL.Patient patient = new Patient(_patientGuid);

			
			// This unit will not satisfy any of the component requirements except WASHED PLT. 
			DataRow drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E0291'");
			DataCreator.CreateBloodUnit(bloodUnitGuid1,Guid.NewGuid(),_patientGuid,(Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
			
			
			// These unit will satisfy all of the component requirements.
			DataRow drProduct = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E3571'");
			DataCreator.CreateBloodUnit(bloodUnitGuid2,Guid.NewGuid(),_patientGuid,(Guid)drProduct[TABLES.BloodProduct.BloodProductGuid], false);
			
		
			// Platelet: This unit will not satisfy any of the component requirements except WASHED RBC and CMV NEG.
			DataRow drProduct2 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E2792'");
			DataCreator.CreateBloodUnit(bloodUnitGuid3,Guid.NewGuid(),_patientGuid,(Guid)drProduct2[TABLES.BloodProduct.BloodProductGuid], false);
			
			// Create all component requirements for the patient
			this.CreateAllCRs(patient);


			ArrayList units = new ArrayList(3);
			units.Add( new BOL.BloodUnit(bloodUnitGuid1) );
			units.Add( new BOL.BloodUnit(bloodUnitGuid2) );
			units.Add( new BOL.BloodUnit(bloodUnitGuid3) );



			Hashtable results = patient.ValidateComponentRequirements(units);
			
			// Three arraylists are returned, one for each unit
			Assert.IsTrue(results.Count == 3);


			// Results.
			ArrayList compResults;

			// 5 requirements not satisfied
			compResults = (ArrayList) results[bloodUnitGuid1];
			Assert.IsTrue(compResults.Count == 5);

			// All requirements satisfied
			compResults = (ArrayList) results[bloodUnitGuid2];
			Assert.IsTrue(compResults.Count == 0);

			// 4 requirements not satisfied
			compResults = (ArrayList) results[bloodUnitGuid3];
			Assert.IsTrue(compResults.Count == 4);


			// Clean up
			this.DeleteAllCRs(patient);
		}


		[Test]
		public void HasActiveTRs_FromConversion_Pass()
		{
			Guid TRGuid = Guid.NewGuid();
			Guid patientGuid = DataCreator.GetRecordGuid("SELECT TOP 1 PatientGuid FROM Patient");

			string insertTransfusionReq = "INSERT INTO PATIENTTRANSFUSIONREQUIREMENT (";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.PatientGuid + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementText + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.CreateDivisionCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.VistaConversionIndicator + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.AntibodyTypeId + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.RecordStatusCode + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateDate + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateUser + ",";
			insertTransfusionReq += TABLES.PatientTransfusionRequirement.LastUpdateFunctionId + ")";
			insertTransfusionReq += "VALUES (";
			insertTransfusionReq += "'" + TRGuid.ToString() + "',";
			insertTransfusionReq += "'" + patientGuid.ToString() + "',";
			insertTransfusionReq += "'Anti-C',";
			insertTransfusionReq += "'AS',";
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertTransfusionReq += "'1',";
			insertTransfusionReq += "'5',";
			insertTransfusionReq += "'C',";  //TR from conversion
			insertTransfusionReq += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertTransfusionReq += "'" + Environment.UserName + "',";
			insertTransfusionReq += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";

			UnitTestUtil.RunSQL(insertTransfusionReq);

			BOL.Patient patient = new Patient( patientGuid );
			
			Assert.IsTrue( patient.HasActiveTRs );
		}




		private void DeleteConflictingData(BOL.Patient patient, 
			BOL.BloodUnit bloodUnit,
			bool bloodUnitAntigen,
			bool patientTransfusionRequirement,
			bool bloodUnitTests)
		{
			if (bloodUnitAntigen) UnitTestUtil.RunSQL("DELETE FROM BloodUnitAntigen WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid.ToString() + "'");
			if (patientTransfusionRequirement) UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patient.PatientGuid.ToString() + "'");
			if (bloodUnitTests) UnitTestUtil.RunSQL("DELETE FROM BloodUnitTest WHERE BloodUnitGuid = '" + bloodUnit.BloodUnitGuid + "'");
		}

		private void SetupBloodUnitTests(Guid bloodUnitGuid, int test, string result)
		{
			Guid guid = Guid.NewGuid();
			DataCreator.InsertBloodUnitTest(guid, bloodUnitGuid, Guid.Empty, Guid.Empty, test, result);
		}

		private void SetupAntigenTr(BOL.Patient patient, Guid bloodUnitGuid)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
					
			// Create new TR (antigen neg for C) 
			BOL.PatientTransfusionRequirement trAntigen = new BOL.PatientTransfusionRequirement(patient, "C", Common.TransfusionRequirementCategoryCode.AntigenNegative, UnitTestConfig.TestDivisionCode);
			DataRow drAntigen = dtInsertTransfusionRequirements.NewRow();

			Guid antigenGuid = Guid.NewGuid();
			drAntigen[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antigenGuid;
			drAntigen[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntigen.TransfusionRequirement;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntigen.TransfusionRequirementCategoryCode );
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDate] = trAntigen.CreateDate;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateUser] = trAntigen.CreateUser;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntigen.CreateDivisionCode;
			drAntigen[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntigen[TABLES.PatientTransfusionRequirement.AntigenTypeId] = 6;
			drAntigen[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntigen[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntigen[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntigen);

			// Create new TR (antigen neg for D) 
			trAntigen = new BOL.PatientTransfusionRequirement(patient, "D", Common.TransfusionRequirementCategoryCode.AntigenNegative, UnitTestConfig.TestDivisionCode);
			drAntigen = dtInsertTransfusionRequirements.NewRow();

			antigenGuid = Guid.NewGuid();
			drAntigen[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antigenGuid;
			drAntigen[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntigen.TransfusionRequirement;
			drAntigen[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntigen.TransfusionRequirementCategoryCode );
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDate] = trAntigen.CreateDate;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateUser] = trAntigen.CreateUser;
			drAntigen[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntigen.CreateDivisionCode;
			drAntigen[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntigen[TABLES.PatientTransfusionRequirement.AntigenTypeId] = 15;
			drAntigen[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntigen[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntigen[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntigen);

			BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
		}

		private void SetupAntibodyTr(BOL.Patient patient, Guid bloodUnitGuid)
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
					
			// Create new TR (antibody for Anti-E) 
			BOL.PatientTransfusionRequirement trAntibody = new BOL.PatientTransfusionRequirement(patient, "Anti-E", Common.TransfusionRequirementCategoryCode.AntibodyIdentified, UnitTestConfig.TestDivisionCode);
			DataRow drAntibody = dtInsertTransfusionRequirements.NewRow();

			Guid antibodyGuid = Guid.NewGuid();
			drAntibody[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antibodyGuid;
			drAntibody[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntibody.TransfusionRequirement;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntibody.TransfusionRequirementCategoryCode );
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDate] = trAntibody.CreateDate;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateUser] = trAntibody.CreateUser;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntibody.CreateDivisionCode;
			drAntibody[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntibody[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 16;
			drAntibody[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntibody[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntibody[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntibody);


			// Create new TR (antibody for Anti-A) - this one should be ignored by the code
			trAntibody = new BOL.PatientTransfusionRequirement(patient, "Anti-A", Common.TransfusionRequirementCategoryCode.AntibodyIdentified, UnitTestConfig.TestDivisionCode);
			drAntibody = dtInsertTransfusionRequirements.NewRow();

			antibodyGuid = Guid.NewGuid();
			drAntibody[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antibodyGuid;
			drAntibody[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntibody.TransfusionRequirement;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntibody.TransfusionRequirementCategoryCode );
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDate] = trAntibody.CreateDate;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateUser] = trAntibody.CreateUser;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntibody.CreateDivisionCode;
			drAntibody[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntibody[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 1;
			drAntibody[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntibody[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntibody[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntibody);

			// Create new TR (antibody for Anti-D) - this one should be ignored by the code
			trAntibody = new BOL.PatientTransfusionRequirement(patient, "Anti-D", Common.TransfusionRequirementCategoryCode.AntibodyIdentified, UnitTestConfig.TestDivisionCode);
			drAntibody = dtInsertTransfusionRequirements.NewRow();

			antibodyGuid = Guid.NewGuid();
			drAntibody[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = antibodyGuid;
			drAntibody[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = trAntibody.TransfusionRequirement;
			drAntibody[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( trAntibody.TransfusionRequirementCategoryCode );
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDate] = trAntibody.CreateDate;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateUser] = trAntibody.CreateUser;
			drAntibody[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = trAntibody.CreateDivisionCode;
			drAntibody[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drAntibody[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 11;
			drAntibody[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drAntibody[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drAntibody[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drAntibody);

			BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
			UnitTestUtil.RunSQL("DELETE FROM BloodUnitAntigen WHERE BloodUnitGuid = '" + bloodUnitGuid.ToString() + "'");
		}

		public void SetupUnitTest(BOL.BloodUnit bloodUnit, string positive, int antigenId)
		{
			UnitTestUtil.RunSQL("INSERT INTO BloodUnitAntigen " +
				"(BloodUnitGUID, AntigenTypeID,	Positive, " +
				"RecordStatusCode, DivisionCode, LastUpdateDate, " +
				"LastUpdateUser, LastUpdateFunctionID ) " +
				"VALUES ('" +
				bloodUnit.BloodUnitGuid + "', " +
				antigenId + ", '" +
				positive + "', " +
				"'A', '" + 
				UnitTestConfig.TestDivisionCode + "' , '" +
				DateTime.Now.ToString() + "', '" + 
				UnitTestConfig.User_Technologist + "', " + 
				(int) Common.UpdateFunction.UnitTests + ")"		
				);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void ValidateAntibodyAntigenRequirementsMet_Fail()
		{
		}

		[Test]
		public void ValidateProductType_Pass()
		{
			Guid unitGuid = (Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid);
			Common.ComponentClass compClass = Common.ComponentClass.RBC;
			
			// RBCs
			String[] productTypesRBC = {"E001","E002","E003","E004","E005","E006","E007","E008","E009","E049","E050","E051","E052","E053","E054"};
			compClass = Common.ComponentClass.RBC;
			for (int i=0; i<productTypesRBC.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesRBC[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(valid);
			}

			// FFP
			string[] productTypesFFP = {"E010","E011","E012","E013"};
			compClass = Common.ComponentClass.FFP;
			for (int i=0; i<productTypesFFP.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesFFP[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(valid);
			}

			// CRYO
			string [] productTypesCRYO = {"E028","E029","E030","E031","E032","E033"};
			compClass = Common.ComponentClass.CRYO;
			for (int i=0; i<productTypesCRYO.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesCRYO[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(valid);
			}

			// PLATELET
			string [] productTypesPLT = {"E020","E021","E022","E023","E024","E025","E026","E027"};
			compClass = Common.ComponentClass.PLT;
			for (int i=0; i<productTypesPLT.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesPLT[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(valid);
			}

			// OTHER
			string [] productTypesOTHER = {"E017","E018","E014","E015","E019","E034","E035","E036","E037","E038","E039","E040","E041","E042","E043","E044","E045"};
			compClass = Common.ComponentClass.Other;
			for (int i=0; i<productTypesOTHER.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesOTHER[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(valid);
			}

			// WB
			string [] productTypesWB = {"E001","E002","E003","E004","E005","E006","E007","E008","E009","E049","E050","E051"};
			compClass = Common.ComponentClass.WB;
			for (int i=0; i<productTypesWB.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesWB[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(valid);
			}
		}

		
		[Test]
		public void ValidateProductType_Fail()
		{
			Guid unitGuid = (Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid);
			Common.ComponentClass compClass = Common.ComponentClass.RBC;
			
			// RBCs
			String[] productTypesRBC = {"E010","E011","E012","E013","E014","E015","E016","E017","E018","E019","E020","E021","E022","E023","E024","E025","E026","E027","E028","E029","E030","E031","E032","E033","E034","E035","E036","E037","E038","E039","E040","E041","E042","E043","E044","E045","E046","E047","E048"};
			compClass = Common.ComponentClass.RBC;
			for (int i=0; i<productTypesRBC.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesRBC[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(!valid);
			}

			// FFP
			string[] productTypesFFP = {"E001","E002","E003","E004","E005","E006","E007","E008","E009","E014","E015","E016","E017","E018","E019","E020","E021","E022","E023","E024","E025","E026","E027","E028","E029","E030","E031","E032","E033","E034","E035","E036","E037","E038","E039","E040","E041","E042","E043","E044","E045","E046","E047","E048","E049","E050","E051"};
			compClass = Common.ComponentClass.FFP;
			for (int i=0; i<productTypesFFP.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesFFP[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(!valid);
			}

			// CRYO
			string [] productTypesCRYO = {"E001","E002","E003","E004","E005","E006","E007","E008","E009","E010","E011","E012","E013","E014","E015","E016","E017","E018","E019","E020","E021","E022","E023","E024","E025","E026","E027","E034","E035","E036","E037","E038","E039","E040","E041","E042","E043","E044","E045","E046","E047","E048","E049","E050","E051"};
			compClass = Common.ComponentClass.CRYO;
			for (int i=0; i<productTypesCRYO.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesCRYO[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(!valid);
			}

			// PLATELET
			string [] productTypesPLT = {"E001","E002","E003","E004","E005","E006","E007","E008","E009","E010","E011","E012","E013","E014","E015","E016","E017","E018","E019","E028","E029","E030","E031","E032","E033","E034","E035","E036","E037","E038","E039","E040","E041","E042","E043","E044","E045","E046","E047","E048","E049","E050","E051"};
			compClass = Common.ComponentClass.PLT;
			for (int i=0; i<productTypesPLT.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesPLT[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(!valid);
			}

			// OTHER
			string [] productTypesOTHER = {"E001","E002","E003","E004","E005","E006","E007","E008","E009","E010","E011","E012","E013","E016","E020","E021","E022","E023","E024","E025","E026","E027","E028","E029","E030","E031","E032","E033","E046","E047","E048","E049","E050","E051"};
			compClass = Common.ComponentClass.Other;
			for (int i=0; i<productTypesOTHER.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesOTHER[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(!valid);
			}

			// WB
			string [] productTypesWB = {"E010","E011","E012","E013","E014","E015","E016","E017","E018","E019","E020","E021","E022","E023","E024","E025","E026","E027","E028","E029","E030","E031","E032","E033","E034","E035","E036","E037","E038","E039","E040","E041","E042","E043","E044","E045","E046","E047","E048"};
			compClass = Common.ComponentClass.WB;
			for (int i=0; i<productTypesWB.Length; i++)
			{
				tmpUnit.ProductType.ProductTypeCode = productTypesWB[i];
				bool valid = _tstPatient.ValidateProductType(compClass, tmpUnit);
				Assert.IsTrue(!valid);
			}
		}


		[Test]
		public void ValidateComponentRequirements_Pass()
		{	
			Guid bloodUnitGuid = Guid.NewGuid();
			BOL.Patient patient = new Patient(_patientGuid);
			// This unit will satisfy all of the component requirements.
			DataRow drProduct = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E3571'");
			DataCreator.CreateBloodUnit(bloodUnitGuid,Guid.NewGuid(),_patientGuid,(Guid)drProduct[TABLES.BloodProduct.BloodProductGuid], false);
			//BOL.BloodUnit bloodUnit = UnitTests.PatientUnitSelection_BOL.CreateBloodUnit("E028", "E3571", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.CRYO);
			// Create all component requirements for the patient
			this.CreateAllCRs(patient);
			Assert.AreEqual(0,patient.ValidateComponentRequirements(bloodUnitGuid).Count,"CRYO_CR_Count");

			// Clean up
			this.DeleteAllCRs(patient);
		}
		[Test]
		public void ValidateComponentRequirements_Fail()
		{
			Guid bloodUnitGuid1 = Guid.NewGuid();
			Guid bloodUnitGuid2 = Guid.NewGuid();

			BOL.Patient patient = new Patient(_patientGuid);
			// This unit will not satisfy any of the component requirements except WASHED PLT. 
			DataRow drProduct1 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E0291'");
			DataCreator.CreateBloodUnit(bloodUnitGuid1,Guid.NewGuid(),_patientGuid,(Guid)drProduct1[TABLES.BloodProduct.BloodProductGuid], false);
			//BOL.BloodUnit bloodUnit = UnitTests.PatientUnitSelection_BOL.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			// Create all component requirements for the patient
			this.CreateAllCRs(patient);
			Assert.AreEqual(5,patient.ValidateComponentRequirements(bloodUnitGuid1).Count,"RBC_CR_Count");

			// Now, try with Platelet:
			// This unit will not satisfy any of the component requirements except WASHED RBC and CMV NEG.
			DataRow drProduct2 = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM dbo.BloodProduct WHERE ProductCode = 'E2792'");
			DataCreator.CreateBloodUnit(bloodUnitGuid2,Guid.NewGuid(),_patientGuid,(Guid)drProduct2[TABLES.BloodProduct.BloodProductGuid], false);
			//bloodUnit = UnitTests.PatientUnitSelection_BOL.CreateBloodUnit("E020", "E2792", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.PLT);
			Assert.AreEqual(4,patient.ValidateComponentRequirements(bloodUnitGuid2).Count,"PLT_CR_Count");

			// Clean up
			this.DeleteAllCRs(patient);
		}

		[Test]
		public void CheckForPositiveAntibodyAntigenRequirements_Pass()
		{
			Guid unitGuid = (Guid)this._dtBloodUnit.Rows[0][TABLES.BloodUnit.BloodUnitGuid];

			Common.TestResult tstValue = this._tstPatient.CheckForPositiveAntibodyAntigenRequirements(unitGuid);
			Assert.IsTrue(true);
		}

		
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void CheckForPositiveAntibodyAntigenRequirements_Fail()
		{
		}

		
		[Test]
		public void GetAllSpecimensForPatient_Pass()
		{
			System.Collections.ArrayList tstValue = this._tstPatient.GetAllSpecimensForPatient(DateTime.Now.AddYears(-1), DateTime.Now);
			Assert.IsTrue(tstValue != null);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetAllSpecimensForPatient_Fail()
		{
			BOL.Patient tmpPatient = new BOL.Patient();
			tmpPatient.GetAllSpecimensForPatient(DateTime.Now, DateTime.Now);
		}
		
		[Test]
		public void GetAllSpecimensForPatientCount_Pass()
		{
			int tstValue = this._tstPatient.GetAllSpecimensForPatientCount();
			Assert.IsTrue(tstValue >= 0);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void GetAllSpecimensForPatientCount_Fail()
		{
			BOL.Patient tmpPatient = new BOL.Patient();
			tmpPatient.GetAllSpecimensForPatientCount();
		}

		[Test] 
		public void GetPatientAboRHHistoryForJustification_Pass()
		{
			// Test A. No justification
			Guid patientGuid = CreatePatientTestData();
			BOL.Patient patient = new Patient(patientGuid);
			ArrayList aboRhHistory = patient.GetPatientAboRHHistoryForJustification();
			//
			Assert.IsTrue(aboRhHistory != null && (aboRhHistory.Count == 0 || aboRhHistory.Count == 0 || aboRhHistory.Count == 0) );
			//
			DeletePatientTestData(patientGuid);

			// Test B.
			aboRhHistory = this._tstPatient.GetPatientAboRHHistoryForJustification();
			Assert.IsTrue(aboRhHistory != null && (aboRhHistory.Count == 0 || aboRhHistory.Count == 2 || aboRhHistory.Count == 4) );

			// See JustifyPatientABORhChange_Pass for more tests
			
		}
		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetPatientAboRHHistoryForJustification_Fail()
		{
		}

		[Test]
		public void JustifyPatientABORhChange_Pass()
		{
			#region Test A. Can Justify: One set of specimen tests, justify based on patient record, "conversion" ABO\Rh = A P

			Guid patientGuid = CreatePatientTestData(new AboRh(Common.ABO.A, Common.RH.Positive));
			//
			#region Create Test Data

			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			//													 
			// First, we need two new Specimen Uids
			DataTable specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			DataColumn [] specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			long maxUid = 999999999999999;
			long minUid = 100000000000000;
			long specimenUid1 = minUid;
			long startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					break;
				}
			}
			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
			query.Remove(0, query.Length);

			#endregion
			
			DataRow drPatient = UnitTestUtil.RunSQLGetDataRow(string.Format(this._patientSelectSQL, patientGuid.ToString()));
			BOL.Patient patient = new Patient(drPatient);			//Loading Patient from datarow to get ArtificialColumns for Justify
			ArrayList aboRhHistory = patient.GetPatientAboRHHistoryForJustification();
			//
			Assert.IsTrue(patient.JustifyPatientABORhChange("A", "N", "Unit Test" ), "Test A-1");
			Assert.IsTrue(aboRhHistory != null && aboRhHistory.Count == 2, "Test A-2");
			//
			// GetPatientAboRHHistoryForJustification Test C. Speciment Test Count == 2, Already been justified
			// Here we test for the case when the results are already justified
			aboRhHistory = patient.GetPatientAboRHHistoryForJustification();
			Assert.IsTrue(aboRhHistory != null && aboRhHistory.Count == 0, "Test A-3");
			//
			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			query.Append("DELETE FROM SPECIMENTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientTreatmentGuid IN (SELECT PatientTreatmentGuid FROM PatientTreatment WHERE PatientGuid ='").Append(patientGuid).Append("') ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");
			UnitTestUtil.RunSQL(query.ToString());

			#endregion
			// All done!

			#endregion

			#region Test B. Can't Justify: One set of specimen tests, justify based on patient record, no "conversion" results for ABO\Rh

			patientGuid = CreatePatientTestData();
			//
			#region Create Test Data

			patientTreatmentTestOneGuid = Guid.NewGuid();
			patientOrderTestOneGuid = Guid.NewGuid();
			orderedTestTestOneGuid = Guid.NewGuid();
			patientSpecimenTestOneGuid = Guid.NewGuid();
			specimenTestAboTestOneGuid = Guid.NewGuid();
			specimenTestRhTestOneGuid = Guid.NewGuid();
			//													 
			// First, we need two new Specimen Uids
			specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			maxUid = 999999999999999;
			minUid = 100000000000000;
			specimenUid1 = minUid;
			startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					break;
				}
			}
			//
			query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
			query.Remove(0, query.Length);

			#endregion
			
			patient = new Patient(patientGuid);
			aboRhHistory = patient.GetPatientAboRHHistoryForJustification();
			//
			// Verify cannot justify in this case
			Assert.IsTrue(aboRhHistory != null && aboRhHistory.Count == 0, "Test B");
			//
			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			query.Append("DELETE FROM SPECIMENTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientTreatmentGuid IN (SELECT PatientTreatmentGuid FROM PatientTreatment WHERE PatientGuid ='").Append(patientGuid).Append("') ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");
			UnitTestUtil.RunSQL(query.ToString());

			#endregion
			// All done!

			#endregion

			#region  Test C. Can Justify: Two sets of specimen tests
			
			patientGuid = CreatePatientTestData();
			//
			#region Create Test Data

			patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientTreatmentTestTwoGuid = Guid.NewGuid();
			patientOrderTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestTwoGuid = Guid.NewGuid();
			orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();
			patientSpecimenTestOneGuid = Guid.NewGuid();
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			specimenTestAboTestOneGuid = Guid.NewGuid();
			specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();				
			//
			// First, we need two new Specimen Uids
			specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			maxUid = 999999999999999;
			minUid = 100000000000000;
			specimenUid1 = minUid;
			long specimenUid2 = minUid;
			startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					startIdx = idx + 1;
					break;
				}
			}
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid2 = idx;
					break;
				}
			}
			//
			query.Remove(0, query.Length);
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid2).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
			query.Remove(0, query.Length);

			#endregion
			//
			drPatient = UnitTestUtil.RunSQLGetDataRow(string.Format(this._patientSelectSQL, patientGuid.ToString()));
			patient = new Patient(drPatient);			//Loading Patient from datarow to get ArtificialColumns for Justify
			patient.GetPatientAboRHHistoryForJustification();
			//
			Assert.IsTrue(patient.JustifyPatientABORhChange("A", "N", "Unit Test" ), "Test C-1");
			//
			// GetPatientAboRHHistoryForJustification Test D. Speciment Test Count == 4, Already been justified
			// Here we test for the case when the results are already justified
			aboRhHistory = patient.GetPatientAboRHHistoryForJustification();
			Assert.IsTrue(aboRhHistory != null && aboRhHistory.Count == 0, "Test C-2");
			//
			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			query.Append("DELETE FROM SPECIMENTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientTreatmentGuid IN (SELECT PatientTreatmentGuid FROM PatientTreatment WHERE PatientGuid ='").Append(patientGuid).Append("') ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");
			UnitTestUtil.RunSQL(query.ToString());

			#endregion
			// All done!

			#endregion

			#region Test D. Can't Justify: Two sets of specimen tests, using the same specimen twice
			patientGuid = CreatePatientTestData();
			//
			#region Create Test Data

			patientTreatmentTestOneGuid = Guid.NewGuid();
			patientTreatmentTestTwoGuid = Guid.NewGuid();
			patientOrderTestOneGuid = Guid.NewGuid();
			patientOrderTestTwoGuid = Guid.NewGuid();
			orderedTestTestOneGuid = Guid.NewGuid();
			orderedTestTestTwoGuid = Guid.NewGuid();
			patientSpecimenTestOneGuid = Guid.NewGuid();
			specimenTestAboTestOneGuid = Guid.NewGuid();
			specimenTestRhTestOneGuid = Guid.NewGuid();
			specimenTestAboTestTwoGuid = Guid.NewGuid();
			specimenTestRhTestTwoGuid = Guid.NewGuid();				
			//
			// First, we need two new Specimen Uids
			specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			maxUid = 999999999999999;
			minUid = 100000000000000;
			specimenUid1 = minUid;
			startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					startIdx = idx + 1;
					break;
				}
			}
			//
			query.Remove(0, query.Length);
			//
			query.Append("BEGIN TRANSACTION T1 ");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			UnitTestUtil.RunSQL(query.ToString());
			query.Remove(0, query.Length);

			#endregion
			//
			drPatient = UnitTestUtil.RunSQLGetDataRow(string.Format(this._patientSelectSQL, patientGuid.ToString()));
			patient = new Patient(drPatient);			//Loading Patient from datarow to get ArtificialColumns for Justify
			aboRhHistory = patient.GetPatientAboRHHistoryForJustification();
			//
			// Verify cannot justify in this case
			Assert.IsTrue(aboRhHistory != null && aboRhHistory.Count == 0, "Test D");
			//
			// Delete test data
			#region Clear Test Data

			query.Append("BEGIN TRANSACTION T2 ");
			query.Append("DELETE FROM SPECIMENTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM SPECIMENTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTSPECIMEN WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestOneGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM ORDEREDTEST WHERE OrderedTestGuid ='").Append(orderedTestTestTwoGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENTORDER WHERE PatientTreatmentGuid IN (SELECT PatientTreatmentGuid FROM PatientTreatment WHERE PatientGuid ='").Append(patientGuid).Append("') ").Append("\n");
			query.Append("DELETE FROM PATIENTTREATMENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("DELETE FROM PATIENT WHERE PatientGuid ='").Append(patientGuid).Append("' ").Append("\n");
			query.Append("COMMIT TRANSACTION T2 ");
			UnitTestUtil.RunSQL(query.ToString());

			#endregion
			// All done!

			#endregion
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void JustifyPatientABORhChange_Fail()
		{
			BOL.Patient patient = new BOL.Patient(Guid.NewGuid());
			this._tstPatient.JustifyPatientABORhChange(string.Empty, string.Empty, string.Empty);
		}

		[Test]
		public void GetTreatingSpecialtiesList_Pass()
		{
			System.Collections.ArrayList tstValue = BOL.Patient.GetTreatingSpecialtiesList();
			Assert.IsTrue(tstValue.Count >= 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void GetTreatingSpecialtiesList_Fail()
		{
		}

		[Test]
		public void BuildDisplayName_Pass()
		{
			string name = BOL.Patient.BuildDisplayName(null,null,null);
			Assert.AreEqual("",name);
			
			name = BOL.Patient.BuildDisplayName(string.Empty.PadRight(30,'X'),"Y",string.Empty.PadRight(30,'Z'));
			Assert.AreEqual("ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ, XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX Y",name);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void BuildDisplayName_Fail()
		{
		}

		[Test]
		public void Search_Pass()
		{
			this._tstPatient.SpecimenUid = "";
			DataTable dtTest = BOL.Patient.Search(this._tstPatient.LastName, this._tstPatient.SSN, this._tstPatient.SpecimenUid, UnitTestConfig.TestDivisionCode);
			Assert.IsTrue(dtTest.Rows.Count > 0);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void Search_Fail()
		{
		}

		[Test]
		public void InsertPatient_Pass()
		{
			Guid patientGuid = Guid.NewGuid();
			DataTable dtTest = SprocHelper.GetInsertPatientSprocTable(patientGuid);

			bool retValue = BOL.Patient.InsertPatient(dtTest.Rows[0], Common.UpdateFunction.UnitTests);

			//If we were a success, delete our new patient
			if (retValue)
			{
				this.DeletePatientTestData(patientGuid);
			}

			Assert.IsTrue(retValue);
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void InsertPatient_Fail()
		{
		}


		[Test]
		public void SearchVistALink_Pass()
		{
			try
			{
				this._tstPatient.SpecimenUid = "";
				VistALinkUnitTestHarness.RequireConnection();
				DataTable dtTest = BOL.Patient.SearchVistALink(this._tstPatient.LastName, this._tstPatient.SSN, this._tstPatient.SpecimenUid);
				Assert.IsTrue(dtTest.Rows.Count >= 0);
			}
			finally
			{
				VistALinkUnitTestHarness.ReleaseConnection();
			}
		}

		[Test]
		[Ignore(StandardIgnoreReasons.NoFailureConditions)]
		public void SearchVistALink_Fail()
		{
		}

		[Test]
		public void SaveSIsAndTRs_Pass()
		{
			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
	
			// Get a patient
			string patientSQL = "SELECT TOP 1 PatientGuid FROM Patient WHERE RecordStatusCode = 'A'";
			DataRow dtPatientRow = UnitTestUtil.RunSQLGetDataRow(patientSQL);
			BOL.Patient patient = new BOL.Patient( (Guid) dtPatientRow[TABLES.Patient.PatientGuid]);

			// Create new SI
			BOL.PatientSpecialInstruction pSI = new BOL.PatientSpecialInstruction(patient, "Testing New SI", UnitTestConfig.TestDivisionCode);

			DataRow dr = dtInsertSpecialInstructions.NewRow();
			dr[TABLES.PatientSpecialInstruction.PatientSpecialInstructionGuid] = Guid.NewGuid();
			dr[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			dr[TABLES.PatientSpecialInstruction.PatientSpecialInstructions] = pSI.PatientSpecialInstructions;
			dr[TABLES.PatientSpecialInstruction.CreateDate] = pSI.CreateDate;
			dr[TABLES.PatientSpecialInstruction.CreateUser] = pSI.CreateUser;
			dr[TABLES.PatientSpecialInstruction.VistaConversionIndicator] = false;
			dr[TABLES.PatientSpecialInstruction.DivisionCode] = UnitTestConfig.TestDivisionCode;
			dr[TABLES.PatientSpecialInstruction.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			dr[TABLES.PatientSpecialInstruction.LastUpdateUser] = UnitTestConfig.User_Technologist;
			dtInsertSpecialInstructions.Rows.Add(dr);
			
			// Create new TR
			BOL.PatientTransfusionRequirement pTR = new BOL.PatientTransfusionRequirement(patient, "Irradiated cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			pTR.ComponentRequirementId = (int) Common.ComponentRequirement.IRRADIATE;
			
			DataRow drTRs = dtInsertTransfusionRequirements.NewRow();

			drTRs[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = Guid.NewGuid();
			drTRs[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = pTR.TransfusionRequirement;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( pTR.TransfusionRequirementCategoryCode );
			drTRs[TABLES.PatientTransfusionRequirement.CreateDate] = pTR.CreateDate;
			drTRs[TABLES.PatientTransfusionRequirement.CreateUser] = pTR.CreateUser;
			drTRs[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = pTR.CreateDivisionCode;
			drTRs[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drTRs[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = pTR.ComponentRequirementId;
			drTRs[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drTRs[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drTRs[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtInsertTransfusionRequirements.Rows.Add(drTRs);

			bool success = BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(success);
		}

		[Test]
		[ExpectedExceptionAttribute(typeof(ArgumentException))]
		public void SaveSIsAndTRs_Fail()
		{
			BOL.Patient.SaveSIsAndTRs(null, null, null, null, Common.UpdateFunction.Invalid);
		}

		#region Patient ABO/Rh

		// The date conditions involved in Patient ABO/Rh are too complex to test 
		// accurately and thoroughly with unit testing.  All conditions will be
		// tested in a test script.
		
		[Test]
		public void GetPatientAboRh_Pass()
		{
			Guid patientGuid = DataCreator.GetRecordGuid("SELECT TOP 1 PatientGuid FROM Patient");

			//This data won't exist until conversion, so we're giving it a boost
			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = 'A', RhFactorCode = 'P' WHERE PatientGuid = '" + patientGuid.ToString() + "'");
		
			BOL.AboRh aboRh = BOL.Patient.GetPatientAboRh(patientGuid);

			Assert.AreEqual(Common.ABO.A,  aboRh.Abo, "ABO");
			Assert.AreEqual(Common.RH.Positive,  aboRh.RH, "RH");
		}

		[Test]
		public void GetPatientAboRh_Fail()
		{
			BOL.AboRh aboRh = BOL.Patient.GetPatientAboRh(Guid.NewGuid());
			Assert.IsTrue(aboRh.Abo == Common.ABO.NA && aboRh.RH == Common.RH.NotProvided);
		}

		#endregion


	

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasAntiDRequirement_Fail()
		{
		}

		[Test]
		public void HasActiveSIs_Pass()
		{
			BOL.Patient patient = new Patient( this._patientSInTRGuid );
			//
			Assert.IsTrue( patient.HasActiveSIs );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasActiveSIs_Fail()
		{
		}

		[Test]
		public void HasActiveTRs_Pass()
		{
			Guid patientGuid = this.CreateSIandTRs();

			BOL.Patient patient = new Patient( patientGuid );
			
			Assert.IsTrue( patient.HasActiveTRs );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void HasActiveTRs_Fail()
		{
		}

		[Test]
		public void IsPatientDeceased_Pass()
		{
			DataRow dr = UnitTestUtil.RunSQLGetDataRow( "SELECT TOP 1 PatientGUID FROM dbo.Patient WHERE PatientDeathDate IS NULL" );
			//
			Guid patientGuid = (Guid)dr[TABLES.Patient.PatientGuid];
			//
			BOL.Patient patient = new Patient( patientGuid );
			//
			Assert.IsTrue( !patient.IsPatientDeceased );
		}

		[Test]
		[Ignore(StandardIgnoreReasons.ReadOnlyProperty)]
		public void IsPatientDeceased_Fail()
		{
		}

		[Test]
		public void GetPatientAboRhJustificationReason_Pass()
		{
			Hashtable htGuids = DataCreator.CreateCrossmatchedBloodUnitTestData();
			
			Guid patientGuid = (Guid)htGuids["patientGuid"];
			Guid stGuid = (Guid)htGuids["specimenTestAboTestOneGuid"];

			UnitTestUtil.RunSQL("UPDATE SpecimenTest SET AboRhChangeIndicator = 1, AboRhChangeJustification = 'UnitTests(8089)' WHERE SpecimenTestGuid = '" + stGuid.ToString() + "'");
			//
			BOL.Patient patient = new Patient( patientGuid );
			
			string reason = patient.GetPatientAboRhJustificationReason();
			Assert.IsTrue( reason != null && (reason.Length > 0) );
		}

		[Test]
		public void GetPatientAboRhJustificationReason_Fail()
		{
			Guid patientGuid = Guid.NewGuid();

			DataTable dtPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			BOL.Patient.InsertPatient(dtPatient.Rows[0], Common.UpdateFunction.UnitTests);
			//
			BOL.Patient patient = new Patient( patientGuid );
			//
			string reason = patient.GetPatientAboRhJustificationReason();
			//
			Assert.IsTrue( reason != null && reason == string.Empty );
			//
			this.DeletePatientTestData(patientGuid);
		}

		#region helper methods

		private Guid CreateSIandTRs()
		{
			new BOL.UnitTests.BOL_PatientTransfusionReaction().SaveImplicatedBloodUnits_Pass();
			Guid patientGuid = DataCreator.GetRecordGuid("SELECT TOP 1 DirectedPatientGuid FROM BloodUnitMedia WHERE DirectedPatientGuid IS NOT NULL ORDER BY LastUpdateDate DESC");

			DataTable dtNewTR = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtOldTR = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);
			DataTable dtNewSI = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtOldSI = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);

			
			DataRow dr1 = dtNewTR.NewRow();
			dr1[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = Guid.NewGuid();
			dr1[TABLES.PatientSpecialInstruction.PatientGuid] = patientGuid;
			dr1[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = "UnitTest(7553)";
			dr1[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = "AS";
			dr1[TABLES.PatientTransfusionRequirement.CreateDate] = DateTime.Now.AddDays(-1);
			dr1[TABLES.PatientTransfusionRequirement.CreateUser] = Common.LogonUser.LogonUserName;
			dr1[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = this.TestDivisionCode;
			dr1[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			dr1[TABLES.PatientTransfusionRequirement.AntibodyTypeId] = 5;		//AntiC
			dr1[TABLES.PatientTransfusionRequirement.RecordStatusCode] = "A";
			dr1[TABLES.PatientTransfusionRequirement.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dr1[TABLES.PatientTransfusion.DivisionCode] = this.TestDivisionCode;
			dtNewTR.Rows.Add(dr1);

			DataRow dr2 = dtNewTR.NewRow();
			dr2[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = Guid.NewGuid();
			dr2[TABLES.PatientSpecialInstruction.PatientGuid] = patientGuid;
			dr2[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = "UnitTest(7553)";
			dr2[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = "CR";
			dr2[TABLES.PatientTransfusionRequirement.CreateDate] = DateTime.Now.AddDays(-1);
			dr2[TABLES.PatientTransfusionRequirement.CreateUser] = Common.LogonUser.LogonUserName;
			dr2[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = this.TestDivisionCode;
			dr2[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			dr2[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = 6;		//CMV Negative Cellular products
			dr2[TABLES.PatientTransfusionRequirement.RecordStatusCode] = "A";
			dr2[TABLES.PatientTransfusionRequirement.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dr2[TABLES.PatientTransfusion.DivisionCode] = this.TestDivisionCode;
			dtNewTR.Rows.Add(dr2);

			DataRow dr3 = dtNewTR.NewRow();
			dr3[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = Guid.NewGuid();
			dr3[TABLES.PatientSpecialInstruction.PatientGuid] = patientGuid;
			dr3[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = "UnitTest(7553)";
			dr3[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = "AN";
			dr3[TABLES.PatientTransfusionRequirement.CreateDate] = DateTime.Now.AddDays(-1);
			dr3[TABLES.PatientTransfusionRequirement.CreateUser] = Common.LogonUser.LogonUserName;
			dr3[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = this.TestDivisionCode;
			dr3[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			dr3[TABLES.PatientTransfusionRequirement.AntigenTypeId] = 6;		//C
			dr3[TABLES.PatientTransfusionRequirement.RecordStatusCode] = "A";
			dr3[TABLES.PatientTransfusionRequirement.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dr3[TABLES.PatientTransfusion.DivisionCode] = this.TestDivisionCode;
			dtNewTR.Rows.Add(dr3);

			//Special Instructions
			DataRow dr4 = dtNewSI.NewRow();
			dr4[TABLES.PatientSpecialInstruction.PatientSpecialInstructionGuid] = Guid.NewGuid();
			dr4[TABLES.PatientSpecialInstruction.PatientGuid] = patientGuid;
			dr4[TABLES.PatientSpecialInstruction.PatientSpecialInstructions] = "UnitTest(7553)";
			dr4[TABLES.PatientSpecialInstruction.CreateDate] = DateTime.Now.AddDays(-1);
			dr4[TABLES.PatientSpecialInstruction.CreateUser] = Common.LogonUser.LogonUserName;
			dr4[TABLES.PatientSpecialInstruction.VistaConversionIndicator] = false;
			dr4[TABLES.PatientSpecialInstruction.RecordStatusCode] = "A";
			dr4[TABLES.PatientSpecialInstruction.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dr4[TABLES.PatientSpecialInstruction.DivisionCode] = this.TestDivisionCode;
			dtNewSI.Rows.Add(dr4);

			//Save the SIs and TRs
			BOL.Patient.SaveSIsAndTRs(dtNewSI, dtOldSI, dtNewTR, dtOldTR, Common.UpdateFunction.UnitTests);

			return(patientGuid);
		}

		/// <summary>
		/// CreateAllCRs
		/// </summary>
		private void CreateAllCRs(BOL.Patient patient)
		{

			DataTable dtInsertSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(false);
			DataTable dtUpdateSpecialInstructions = BOL.PatientSpecialInstruction.InitializeSpecialInstructionsTable(true);
			DataTable dtInsertTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(false);
			DataTable dtUpdateTransfusionRequirements = BOL.PatientTransfusionRequirement.InitializeTransfusionRequirementsTable(true);

			UnitTestUtil.RunSQL("DELETE FROM PatientTransfusionRequirement WHERE PatientGuid = '" + patient.PatientGuid + "'");
			
			// Irradiated
			BOL.PatientTransfusionRequirement pTR = new BOL.PatientTransfusionRequirement(patient, "Irradiated cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			Common.ComponentRequirement req = Common.ComponentRequirement.IRRADIATE;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Leuko
			pTR = new BOL.PatientTransfusionRequirement(patient, "Leuko-reduced cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.LEUKORED;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Washed RBC
			pTR = new BOL.PatientTransfusionRequirement(patient, "Washed RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.WASHEDRBC;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Washed PLT
			pTR = new BOL.PatientTransfusionRequirement(patient, "Washed PLT products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.WASHEDPLT;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// Sickle cell
			pTR = new BOL.PatientTransfusionRequirement(patient, "Sickle Cell Negative RBC products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.SCNEG;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);

			// CMV Neg
			pTR = new BOL.PatientTransfusionRequirement(patient, "CMV Negative cellular products", Common.TransfusionRequirementCategoryCode.ComponentRequirement, UnitTestConfig.TestDivisionCode);
			req = Common.ComponentRequirement.CMVNEG;
			AddDataRow(dtInsertTransfusionRequirements, req, pTR, patient);
			
			bool success = BOL.Patient.SaveSIsAndTRs(dtInsertSpecialInstructions, dtUpdateSpecialInstructions, dtInsertTransfusionRequirements, dtUpdateTransfusionRequirements, Common.UpdateFunction.UnitTests);
			Assert.IsTrue(success);
		}

		/// <summary>
		/// AddDataRow
		/// </summary>
		private void AddDataRow(DataTable dtCrs, Common.ComponentRequirement req, BOL.PatientTransfusionRequirement pTR, BOL.Patient patient)
		{
			DataRow drTRs = dtCrs.NewRow();
			drTRs[TABLES.PatientTransfusionRequirement.PatientTransfusionRequirementGuid] = Guid.NewGuid();
			drTRs[TABLES.PatientSpecialInstruction.PatientGuid] = patient.PatientGuid;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementText] = pTR.TransfusionRequirement;
			drTRs[TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode] = Common.Utility.GetTransfusionCategoryCodeFromEnum( pTR.TransfusionRequirementCategoryCode );
			drTRs[TABLES.PatientTransfusionRequirement.CreateDate] = pTR.CreateDate;
			drTRs[TABLES.PatientTransfusionRequirement.CreateUser] = pTR.CreateUser;
			drTRs[TABLES.PatientTransfusionRequirement.CreateDivisionCode] = pTR.CreateDivisionCode;
			drTRs[TABLES.PatientTransfusionRequirement.VistaConversionIndicator] = false;
			drTRs[TABLES.PatientTransfusionRequirement.ComponentRequirementId] = (int) req;
			drTRs[TABLES.PatientTransfusionRequirement.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			drTRs[TABLES.PatientTransfusionRequirement.LastUpdateUser] = UnitTestConfig.User_Technologist;
			drTRs[TABLES.PatientTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			dtCrs.Rows.Add(drTRs);
		}

		/// <summary>
		/// DeleteAllCRs
		/// </summary>
		private void DeleteAllCRs(BOL.Patient patient)
		{
			string sql = "DELETE FROM " + TABLES.PatientTransfusionRequirement.TableName + " WHERE " +
				TABLES.PatientTransfusionRequirement.PatientGuid + " = '" + patient.PatientGuid + "' AND " +
				TABLES.PatientTransfusionRequirement.TransfusionRequirementCategoryCode + " = 'CR'";
			UnitTestUtil.RunSQL(sql);
		}

		/// <summary>
		/// CreatePatientTestData
		/// </summary>
		/// <returns>Guid</returns>
		private Guid CreatePatientTestData()
		{
			Guid patientGuid = Guid.NewGuid();

			DataTable dtInsPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			bool retValue = BOL.Patient.InsertPatient(dtInsPatient.Rows[0], Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "PatientInsert");

			return patientGuid;
		}

		/// <summary>
		/// Inserts merge test data into PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		public Guid InsertMergePatientTestData(Patient mergeFromPatient, Patient mergeToPatient)
		{		
			StringBuilder query = new StringBuilder();
			// 
			Guid patientChangeGuid = Guid.NewGuid();
			//
			query.Remove(0, query.Length);
			query.Append("INSERT INTO PatientChange (PatientChangeGuid, FromPatientGuid, ToPatientGuid, MergeFromVistaPatientId, ");
			query.Append("MergeToVistaPatientId, PatientChangeStatusCode, NotificationReceivedDate, LastUpdateFunctionId, LastUpdateUser, LastUpdateDate)");
			query.Append("VALUES ('");
			query.Append(patientChangeGuid).Append("', '");
			query.Append(mergeFromPatient.PatientGuid).Append("', '");
			query.Append(mergeToPatient.PatientGuid).Append("', '");
			query.Append(mergeFromPatient.VistaPatientId).Append("', '");
			query.Append(mergeToPatient.VistaPatientId).Append("', 'P', GetUtcDate(), 0, 'UNITTEST', GetUtcDate())");
			//
			int rowCount = UnitTestUtil.RunSQL(query.ToString());
			//
			Assert.IsTrue(rowCount > 0, "UNABLE TO INSERT PATIENT CHANGE DATA!");
			//
			return patientChangeGuid;
		}

		/// <summary>
		/// CreatePatientTestData
		/// </summary>
		/// <returns>Guid</returns>
		private Guid CreatePatientTestData(BOL.AboRh aboRh)
		{
			Guid patientGuid = Guid.NewGuid();

			DataTable dtInsPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			bool retValue = BOL.Patient.InsertPatient(dtInsPatient.Rows[0], Common.UpdateFunction.UnitTests);
			Assert.AreEqual(true, retValue, "PatientInsert");

			UnitTestUtil.RunSQL("UPDATE Patient SET BloodTypeCode = '" + aboRh.Abo.ToString() + "', RhFactorCode = '" + aboRh.RHFactorCode + "' WHERE PatientGuid = '" + patientGuid.ToString() + "'");

			return patientGuid;
		}

		/// <summary>
		/// Clears merge test data from PatientChange
		/// </summary>
		/// <returns>Row change count</returns>
		public void ClearMergePatientTestData(Guid patientChangeGuid)
		{
			UnitTestUtil.RunSQL(string.Concat("DELETE FROM PatientChange WHERE PatientChangeGuid = '", patientChangeGuid, "'"));
		}

		/// <summary>
		/// DeletePatientTestData
		/// </summary>
		private void DeletePatientTestData(Guid patientGuid)
		{
			UnitTestUtil.RunSQL("DELETE FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'");
			UnitTestUtil.RunSQL("DELETE FROM Patient WHERE PatientGuid ='" + patientGuid.ToString() + "'");
		}

		#endregion
	}
}

#endif





